/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.com;

import java.time.Clock;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.Assert;
import org.junit.Test;
import org.neo4j.com.ResourcePool;
import org.neo4j.time.Clocks;
import org.neo4j.time.FakeClock;

public class ResourcePoolTest {
    private static final int TIMEOUT_MILLIS = 100;
    private static final int TIMEOUT_EXCEED_MILLIS = 110;

    @Test
    public void shouldNotReuseBrokenInstances() throws Exception {
        ResourcePool<Something> pool = new ResourcePool<Something>(5){

            protected Something create() {
                return new Something();
            }

            protected boolean isAlive(Something resource) {
                return !resource.closed;
            }
        };
        Something somethingFirst = (Something)pool.acquire();
        somethingFirst.doStuff();
        pool.release();
        Something something = (Something)pool.acquire();
        Assert.assertEquals((Object)somethingFirst, (Object)something);
        something.doStuff();
        something.close();
        pool.release();
        Something somethingElse = (Something)pool.acquire();
        Assert.assertFalse((something == somethingElse ? 1 : 0) != 0);
        somethingElse.doStuff();
    }

    @Test
    public void shouldTimeoutGracefully() {
        FakeClock clock = this.getFakeClocks();
        ResourcePool.CheckStrategy.TimeoutCheckStrategy timeStrategy = new ResourcePool.CheckStrategy.TimeoutCheckStrategy(100L, (Clock)clock);
        while (clock.millis() <= 100L) {
            Assert.assertFalse((boolean)timeStrategy.shouldCheck());
            clock.forward(10L, TimeUnit.MILLISECONDS);
        }
        Assert.assertTrue((boolean)timeStrategy.shouldCheck());
        clock.forward(1L, TimeUnit.MILLISECONDS);
        Assert.assertFalse((boolean)timeStrategy.shouldCheck());
    }

    @Test
    public void shouldBuildUpGracefullyUntilReachedMinPoolSize() throws InterruptedException {
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = this.getFakeClocks();
        SomethingResourcePool pool = this.getResourcePool(stateMonitor, clock, 5);
        this.acquireFromPool(pool, 5);
        Assert.assertEquals((long)-1L, (long)stateMonitor.currentPeakSize.get());
        Assert.assertEquals((long)-1L, (long)stateMonitor.targetSize.get());
        Assert.assertEquals((long)0L, (long)stateMonitor.disposed.get());
    }

    @Test
    public void shouldBuildUpGracefullyWhilePassingMinPoolSizeBeforeTimerRings() throws InterruptedException {
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = this.getFakeClocks();
        SomethingResourcePool pool = this.getResourcePool(stateMonitor, clock, 5);
        this.acquireFromPool(pool, 15);
        Assert.assertEquals((long)-1L, (long)stateMonitor.currentPeakSize.get());
        Assert.assertEquals((long)15L, (long)stateMonitor.created.get());
        Assert.assertEquals((long)-1L, (long)stateMonitor.targetSize.get());
        Assert.assertEquals((long)0L, (long)stateMonitor.disposed.get());
    }

    @Test
    public void shouldUpdateTargetSizeWhenSpikesOccur() throws Exception {
        int poolMinSize = 5;
        int poolMaxSize = 10;
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = this.getFakeClocks();
        SomethingResourcePool pool = this.getResourcePool(stateMonitor, clock, 5);
        List<ResourceHolder> holders = this.acquireFromPool(pool, 10);
        this.exceedTimeout(clock);
        holders.addAll(this.acquireFromPool(pool, 1));
        Assert.assertEquals((long)11L, (long)stateMonitor.currentPeakSize.get());
        Assert.assertEquals((long)11L, (long)stateMonitor.targetSize.get());
        for (ResourceHolder holder : holders) {
            holder.end();
        }
    }

    @Test
    public void shouldKeepSmallPeakAndNeverDisposeIfAcquireAndReleaseContinuously() throws Exception {
        boolean poolMinSize = true;
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = this.getFakeClocks();
        SomethingResourcePool pool = this.getResourcePool(stateMonitor, clock, 1);
        for (int i = 0; i < 200; ++i) {
            List<ResourceHolder> newOnes = this.acquireFromPool(pool, 1);
            CountDownLatch release = new CountDownLatch(newOnes.size());
            for (ResourceHolder newOne : newOnes) {
                newOne.release(release);
            }
            release.await();
        }
        Assert.assertEquals((long)-1L, (long)stateMonitor.currentPeakSize.get());
        Assert.assertEquals((long)1L, (long)stateMonitor.created.get());
        Assert.assertEquals((long)0L, (long)stateMonitor.disposed.get());
    }

    @Test
    public void shouldSlowlyReduceTheNumberOfResourcesInThePoolWhenResourcesAreReleased() throws Exception {
        int poolMinSize = 50;
        int poolMaxSize = 200;
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = this.getFakeClocks();
        SomethingResourcePool pool = this.getResourcePool(stateMonitor, clock, 50);
        this.acquireResourcesAndExceedTimeout(pool, clock, 200);
        this.exceedTimeout(clock);
        for (int i = 0; i < 200; ++i) {
            this.acquireFromPool(pool, 1).get(0).release();
        }
        Assert.assertEquals((long)1L, (long)stateMonitor.currentPeakSize.get());
        Assert.assertEquals((long)50L, (long)stateMonitor.targetSize.get());
        Assert.assertEquals((long)50L, (long)pool.unusedSize());
        Assert.assertEquals((long)151L, (long)stateMonitor.disposed.get());
    }

    @Test
    public void shouldMaintainPoolHigherThenMinSizeWhenPeekUsagePasses() throws Exception {
        int poolMinSize = 50;
        int poolMaxSize = 200;
        int afterPeekPoolSize = 90;
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = this.getFakeClocks();
        SomethingResourcePool pool = this.getResourcePool(stateMonitor, clock, 50);
        this.acquireResourcesAndExceedTimeout(pool, clock, 200);
        this.exceedTimeout(clock);
        for (int i = 0; i < 10; ++i) {
            CountDownLatch release = new CountDownLatch(90);
            for (ResourceHolder holder : this.acquireFromPool(pool, 90)) {
                holder.release(release);
            }
            release.await();
            this.exceedTimeout(clock);
        }
        Assert.assertEquals((long)90L, (long)stateMonitor.currentPeakSize.get());
        Assert.assertEquals((long)90L, (long)stateMonitor.targetSize.get());
        Assert.assertEquals((long)90L, (long)pool.unusedSize());
        Assert.assertThat((Object)stateMonitor.disposed.get(), (Matcher)Matchers.greaterThanOrEqualTo((Comparable)Integer.valueOf(111)));
    }

    @Test
    public void shouldReclaimAndRecreateWhenUsageGoesDownBetweenSpikes() throws Exception {
        int poolMinSize = 50;
        int bellowPoolMinSize = 10;
        int poolMaxSize = 200;
        StatefulMonitor stateMonitor = new StatefulMonitor();
        FakeClock clock = this.getFakeClocks();
        SomethingResourcePool pool = this.getResourcePool(stateMonitor, clock, 50);
        this.acquireResourcesAndExceedTimeout(pool, clock, 200);
        this.exceedTimeout(clock);
        for (int i = 0; i < 30; ++i) {
            CountDownLatch release = new CountDownLatch(10);
            for (ResourceHolder holder : this.acquireFromPool(pool, 10)) {
                holder.release(release);
            }
            release.await();
            this.exceedTimeout(clock);
        }
        Assert.assertTrue((String)String.valueOf(stateMonitor.currentPeakSize.get()), (stateMonitor.currentPeakSize.get() <= 10 ? 1 : 0) != 0);
        Assert.assertEquals((long)50L, (long)stateMonitor.targetSize.get());
        Assert.assertEquals((long)50L, (long)pool.unusedSize());
        Assert.assertEquals((long)151L, (long)stateMonitor.disposed.get());
        stateMonitor.created.set(0);
        stateMonitor.disposed.set(0);
        this.acquireResourcesAndExceedTimeout(pool, clock, 200);
        Assert.assertEquals((long)151L, (long)stateMonitor.created.get());
        Assert.assertEquals((long)0L, (long)stateMonitor.disposed.get());
    }

    private void exceedTimeout(FakeClock clock) {
        clock.forward(110L, TimeUnit.MILLISECONDS);
    }

    private FakeClock getFakeClocks() {
        return Clocks.fakeClock();
    }

    private void acquireResourcesAndExceedTimeout(ResourcePool<Something> pool, FakeClock clock, int resourcesToAcquire) throws InterruptedException {
        LinkedList<ResourceHolder> holders = new LinkedList<ResourceHolder>();
        holders.addAll(this.acquireFromPool(pool, resourcesToAcquire));
        this.exceedTimeout(clock);
        holders.addAll(this.acquireFromPool(pool, 1));
        for (ResourceHolder holder : holders) {
            holder.release();
        }
    }

    private SomethingResourcePool getResourcePool(StatefulMonitor stateMonitor, FakeClock clock, int minSize) {
        ResourcePool.CheckStrategy.TimeoutCheckStrategy timeoutCheckStrategy = new ResourcePool.CheckStrategy.TimeoutCheckStrategy(100L, (Clock)clock);
        return new SomethingResourcePool(minSize, (ResourcePool.CheckStrategy)timeoutCheckStrategy, stateMonitor);
    }

    private List<ResourceHolder> acquireFromPool(ResourcePool pool, int resourcesToAcquire) throws InterruptedException {
        LinkedList<ResourceHolder> acquirers = new LinkedList<ResourceHolder>();
        CountDownLatch latch = new CountDownLatch(resourcesToAcquire);
        for (int i = 0; i < resourcesToAcquire; ++i) {
            ResourceHolder holder = new ResourceHolder(pool, latch);
            Thread t = new Thread(holder);
            acquirers.add(holder);
            t.start();
        }
        latch.await();
        return acquirers;
    }

    private static class Something {
        private boolean closed;

        private Something() {
        }

        public void doStuff() throws Exception {
            if (this.closed) {
                throw new Exception("Closed");
            }
        }

        public void close() {
            this.closed = true;
        }
    }

    private class StatefulMonitor
    implements ResourcePool.Monitor<Something> {
        public AtomicInteger currentPeakSize = new AtomicInteger(-1);
        public AtomicInteger targetSize = new AtomicInteger(-1);
        public AtomicInteger created = new AtomicInteger(0);
        public AtomicInteger acquired = new AtomicInteger(0);
        public AtomicInteger disposed = new AtomicInteger(0);

        private StatefulMonitor() {
        }

        public void updatedCurrentPeakSize(int currentPeakSize) {
            this.currentPeakSize.set(currentPeakSize);
        }

        public void updatedTargetSize(int targetSize) {
            this.targetSize.set(targetSize);
        }

        public void created(Something something) {
            this.created.incrementAndGet();
        }

        public void acquired(Something something) {
            this.acquired.incrementAndGet();
        }

        public void disposed(Something something) {
            this.disposed.incrementAndGet();
        }
    }

    private class ResourceHolder
    implements Runnable {
        private final Semaphore latch = new Semaphore(0);
        private final CountDownLatch released = new CountDownLatch(1);
        private final CountDownLatch onAcquire;
        private final ResourcePool pool;
        private final AtomicBoolean release = new AtomicBoolean();
        private volatile Thread runner;

        private ResourceHolder(ResourcePool pool, CountDownLatch onAcquire) {
            this.pool = pool;
            this.onAcquire = onAcquire;
        }

        @Override
        public void run() {
            this.runner = Thread.currentThread();
            try {
                this.pool.acquire();
                this.onAcquire.countDown();
                try {
                    this.latch.acquire();
                }
                catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
                if (this.release.get()) {
                    this.pool.release();
                    this.released.countDown();
                }
            }
            catch (Throwable e) {
                e.printStackTrace();
            }
        }

        public void release() {
            this.release.set(true);
            this.latch.release();
            try {
                Thread runner;
                this.released.await();
                while ((runner = this.runner) == null) {
                }
                runner.join();
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        public void release(CountDownLatch releaseLatch) {
            this.release();
            releaseLatch.countDown();
        }

        public void end() {
            this.release.set(false);
            this.latch.release();
        }
    }

    private static class SomethingResourcePool
    extends ResourcePool<Something> {
        SomethingResourcePool(int minSize, ResourcePool.CheckStrategy checkStrategy, StatefulMonitor stateMonitor) {
            super(minSize, checkStrategy, (ResourcePool.Monitor)stateMonitor);
        }

        protected Something create() {
            return new Something();
        }

        protected boolean isAlive(Something resource) {
            return !resource.closed;
        }

        public int unusedSize() {
            return this.unused.size();
        }
    }
}

