package com.aliyun.sdk.gateway.pop.interceptor.httpRequest;

import com.aliyun.core.http.*;
import com.aliyun.sdk.gateway.pop.auth.RoaSignatureComposer;
import com.aliyun.sdk.gateway.pop.auth.SignatureVersion;
import com.aliyun.auth.credentials.ICredential;
import com.aliyun.auth.signature.SignerParams;
import com.aliyun.auth.credentials.provider.AnonymousCredentialProvider;
import com.aliyun.sdk.gateway.pop.auth.signer.PopSigner;
import com.aliyun.core.logging.ClientLogger;
import com.aliyun.core.utils.AttributeMap;
import com.aliyun.core.utils.ParseUtil;
import com.aliyun.core.utils.StringUtils;
import com.aliyun.sdk.gateway.pop.exception.PopClientException;
import darabonba.core.RequestStyle;
import darabonba.core.TeaConfiguration;
import darabonba.core.TeaPair;
import darabonba.core.TeaRequest;
import darabonba.core.client.ClientConfiguration;
import darabonba.core.client.ClientOption;
import darabonba.core.interceptor.HttpRequestInterceptor;
import darabonba.core.interceptor.InterceptorContext;
import darabonba.core.utils.CommonUtil;
import darabonba.core.utils.ModelUtil;

import java.io.UnsupportedEncodingException;
import java.util.Map;
import java.util.Optional;

public class RoaHttpReqInterceptor implements HttpRequestInterceptor {
    private final ClientLogger logger = new ClientLogger(RoaHttpReqInterceptor.class);

    @Override
    public HttpRequest modifyHttpRequest(InterceptorContext context, AttributeMap attributes) {
        TeaRequest request = context.teaRequest();
        HttpRequest httpRequest = context.httpRequest();
        if (!request.style().equals(RequestStyle.ROA)
                && !request.style().equals(RequestStyle.RESTFUL)) {
            return httpRequest;
        }
        TeaConfiguration configuration = context.configuration();
        ClientConfiguration clientConfiguration = configuration.clientConfiguration();
        PopSigner signer = (PopSigner) clientConfiguration.option(ClientOption.SIGNER);
        if (signer.getSignerVersion() != SignatureVersion.V1) {
            return httpRequest;
        }
        HttpMethod method = Optional.ofNullable(configuration.method()).orElseGet(request::method);
        Map<String, String> headers = CommonUtil.merge(String.class, CommonUtil.buildMap(
                new TeaPair("date", CommonUtil.getDateUTCString()),
                new TeaPair("x-acs-signature-nonce", CommonUtil.getNonce())
                ),
                request.headers().toMap()
        );
        String body = null;
        if (!CommonUtil.isUnset(request.body())) {
            if (request.reqBodyType().equals(BodyType.JSON)) {
                if (request.body() instanceof byte[]) {
                    try {
                        body = new String((byte[]) request.body(), "UTF-8");
                    } catch (UnsupportedEncodingException e) {
                        throw new PopClientException(e);
                    }
                } else {
                    body = ParseUtil.toJSONString(request.body());
                }
                headers.put("content-type", FormatType.JSON);
            } else {
                if (request.body() instanceof byte[]) {
                    try {
                        body = new String((byte[]) request.body(), "UTF-8");
                    } catch (UnsupportedEncodingException e) {
                        throw new PopClientException(e);
                    }
                } else {
                    body = ModelUtil.toFormString(ModelUtil.query(CommonUtil.assertAsMap(request.body())));
                }
                headers.put("content-type", FormatType.FORM);
            }
        }
        Map<String, String> query = request.query();
        if (!(configuration.credentialProvider() instanceof AnonymousCredentialProvider)) {
            ICredential credential = configuration.credentialProvider().getCredentials();
            String accessKeyId = credential.accessKeyId();
            String securityToken = credential.securityToken();
            if (!StringUtils.isEmpty(securityToken)) {
                headers.put("x-acs-accesskey-id", accessKeyId);
                headers.put("x-acs-security-token", securityToken);
            }
            headers.putAll(CommonUtil.buildMap(
                    new TeaPair("x-acs-signature-method", signer.getSignerName().algorithmName()),
                    new TeaPair("x-acs-signature-version", "1.0")));
            String strToSign = RoaSignatureComposer.composeStringToSign(method, query, headers, request.pathname());
            SignerParams params = SignerParams.create();
            params.setCredentials(credential);
            headers.put("authorization", "acs " + accessKeyId + ":" + signer.signString(strToSign, params));
        }
        HttpHeaders httpHeaders = new HttpHeaders(headers);
        httpRequest = new HttpRequest(method,
                ModelUtil.composeUrl(configuration.endpoint(), query, configuration.protocol(), request.pathname()));
        httpRequest.setHeaders(httpHeaders);
        if (!CommonUtil.isUnset(request.body())) {
            httpRequest.setBody(body);
        }
        return httpRequest;
    }
}
