/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.graphdb;

import java.util.concurrent.TimeUnit;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.hamcrest.core.IsNot;
import org.hamcrest.core.IsNull;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.neo4j.graphdb.ConstraintViolationException;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.schema.ConstraintDefinition;
import org.neo4j.graphdb.schema.ConstraintType;
import org.neo4j.graphdb.schema.IndexDefinition;
import org.neo4j.graphdb.schema.Schema;
import org.neo4j.helpers.collection.Iterables;
import org.neo4j.helpers.collection.Iterators;
import org.neo4j.test.mockito.matcher.Neo4jMatchers;
import org.neo4j.test.rule.ImpermanentDatabaseRule;

public class SchemaAcceptanceTest {
    @Rule
    public ImpermanentDatabaseRule dbRule = new ImpermanentDatabaseRule();
    private GraphDatabaseService db;
    private Label label = Labels.MY_LABEL;
    private String propertyKey = "my_property_key";
    private String secondPropertyKey = "my_second_property_key";

    @Before
    public void init() {
        this.db = this.dbRule.getGraphDatabaseAPI();
    }

    @Test
    public void addingAnIndexingRuleShouldSucceed() {
        IndexDefinition index = Neo4jMatchers.createIndex(this.db, this.label, this.propertyKey);
        MatcherAssert.assertThat(Neo4jMatchers.getIndexes(this.db, this.label), Neo4jMatchers.containsOnly(index));
    }

    @Test
    public void addingACompositeIndexingRuleShouldSucceed() {
        IndexDefinition index = Neo4jMatchers.createIndex(this.db, this.label, this.propertyKey, this.secondPropertyKey);
        MatcherAssert.assertThat(Neo4jMatchers.getIndexes(this.db, this.label), Neo4jMatchers.containsOnly(index));
    }

    @Test
    public void addingAnIndexingRuleInNestedTxShouldSucceed() {
        IndexDefinition index;
        IndexDefinition indexDef;
        try (Transaction tx = this.db.beginTx();){
            try (Transaction nestedTransaction = this.db.beginTx();){
                indexDef = this.db.schema().indexFor(this.label).on(this.propertyKey).create();
                nestedTransaction.success();
            }
            index = indexDef;
            tx.success();
        }
        Neo4jMatchers.waitForIndex(this.db, indexDef);
        MatcherAssert.assertThat(Neo4jMatchers.getIndexes(this.db, this.label), Neo4jMatchers.containsOnly(index));
    }

    @Test
    public void shouldThrowConstraintViolationIfAskedToIndexSamePropertyAndLabelTwiceInSameTx() {
        try (Transaction tx = this.db.beginTx();){
            Schema schema = this.db.schema();
            schema.indexFor(this.label).on(this.propertyKey).create();
            try {
                schema.indexFor(this.label).on(this.propertyKey).create();
                Assert.fail((String)"Should not have validated");
            }
            catch (ConstraintViolationException e) {
                Assert.assertEquals((Object)"There already exists an index for label 'MY_LABEL' on property 'my_property_key'.", (Object)e.getMessage());
            }
            tx.success();
        }
    }

    @Test
    public void shouldThrowConstraintViolationIfAskedToIndexPropertyThatIsAlreadyIndexed() {
        Schema schema;
        try (Transaction tx = this.db.beginTx();){
            schema = this.db.schema();
            schema.indexFor(this.label).on(this.propertyKey).create();
            tx.success();
        }
        ConstraintViolationException caught = null;
        try (Transaction tx = this.db.beginTx();){
            schema.indexFor(this.label).on(this.propertyKey).create();
            tx.success();
        }
        catch (ConstraintViolationException e) {
            caught = e;
        }
        MatcherAssert.assertThat((Object)((Object)caught), (Matcher)IsNot.not((Matcher)IsNull.nullValue()));
    }

    @Test
    public void shouldThrowConstraintViolationIfAskedToCreateCompoundConstraint() {
        try (Transaction tx = this.db.beginTx();){
            Schema schema = this.db.schema();
            schema.constraintFor(this.label).assertPropertyIsUnique("my_property_key").assertPropertyIsUnique("other_property").create();
            tx.success();
            Assert.fail((String)"Should not be able to create constraint on multiple propertyKey keys");
        }
        catch (UnsupportedOperationException e) {
            MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"can only create one unique constraint"));
        }
    }

    @Test
    public void droppingExistingIndexRuleShouldSucceed() {
        IndexDefinition index = Neo4jMatchers.createIndex(this.db, this.label, this.propertyKey);
        this.dropIndex(index);
        MatcherAssert.assertThat(Neo4jMatchers.getIndexes(this.db, this.label), Neo4jMatchers.isEmpty());
    }

    @Test
    public void droppingAnUnexistingIndexShouldGiveHelpfulExceptionInSameTransaction() {
        IndexDefinition index = Neo4jMatchers.createIndex(this.db, this.label, this.propertyKey);
        try (Transaction tx = this.db.beginTx();){
            index.drop();
            try {
                index.drop();
                Assert.fail((String)"Should not be able to drop index twice");
            }
            catch (ConstraintViolationException e) {
                MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"No index was found for :MY_LABEL(my_property_key)."));
            }
            tx.success();
        }
        MatcherAssert.assertThat((String)"Index should have been deleted", Neo4jMatchers.getIndexes(this.db, this.label), (Matcher)IsNot.not(Neo4jMatchers.contains(index)));
    }

    @Test
    public void droppingAnUnexistingIndexShouldGiveHelpfulExceptionInSeparateTransactions() {
        IndexDefinition index = Neo4jMatchers.createIndex(this.db, this.label, this.propertyKey);
        this.dropIndex(index);
        try {
            this.dropIndex(index);
            Assert.fail((String)"Should not be able to drop index twice");
        }
        catch (ConstraintViolationException e) {
            MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"No index was found for :MY_LABEL(my_property_key)."));
        }
        MatcherAssert.assertThat((String)"Index should have been deleted", Neo4jMatchers.getIndexes(this.db, this.label), (Matcher)IsNot.not(Neo4jMatchers.contains(index)));
    }

    @Test
    public void awaitingIndexComingOnlineWorks() {
        IndexDefinition index = Neo4jMatchers.createIndex(this.db, this.label, this.propertyKey);
        try (Transaction tx = this.db.beginTx();){
            this.db.schema().awaitIndexOnline(index, 1L, TimeUnit.MINUTES);
            Assert.assertEquals((Object)Schema.IndexState.ONLINE, (Object)this.db.schema().getIndexState(index));
        }
    }

    @Test
    public void awaitingAllIndexesComingOnlineWorks() {
        IndexDefinition index = Neo4jMatchers.createIndex(this.db, this.label, this.propertyKey);
        Neo4jMatchers.createIndex(this.db, this.label, "other_property");
        Neo4jMatchers.waitForIndex(this.db, index);
        try (Transaction tx = this.db.beginTx();){
            this.db.schema().awaitIndexesOnline(1L, TimeUnit.MINUTES);
            Assert.assertEquals((Object)Schema.IndexState.ONLINE, (Object)this.db.schema().getIndexState(index));
        }
    }

    @Test
    public void shouldPopulateIndex() {
        Node node = this.createNode(this.db, this.propertyKey, "Neo", this.label);
        IndexDefinition index = Neo4jMatchers.createIndex(this.db, this.label, this.propertyKey);
        Neo4jMatchers.waitForIndex(this.db, index);
        MatcherAssert.assertThat(Neo4jMatchers.findNodesByLabelAndProperty(this.label, this.propertyKey, "Neo", this.db), Neo4jMatchers.containsOnly(node));
    }

    @Test
    public void shouldRecreateDroppedIndex() {
        Node node = this.createNode(this.db, this.propertyKey, "Neo", this.label);
        IndexDefinition index = Neo4jMatchers.createIndex(this.db, this.label, this.propertyKey);
        Neo4jMatchers.waitForIndex(this.db, index);
        this.dropIndex(index);
        Neo4jMatchers.createIndex(this.db, this.label, this.propertyKey);
        Neo4jMatchers.waitForIndex(this.db, index);
        MatcherAssert.assertThat(Neo4jMatchers.getIndexes(this.db, this.label), Neo4jMatchers.contains(index));
        MatcherAssert.assertThat(Neo4jMatchers.findNodesByLabelAndProperty(this.label, this.propertyKey, "Neo", this.db), Neo4jMatchers.containsOnly(node));
    }

    @Test
    public void shouldCreateUniquenessConstraint() {
        ConstraintDefinition constraint = this.createUniquenessConstraint(this.label, this.propertyKey);
        try (Transaction tx = this.db.beginTx();){
            Assert.assertEquals((Object)ConstraintType.UNIQUENESS, (Object)constraint.getConstraintType());
            Assert.assertEquals((Object)this.label.name(), (Object)constraint.getLabel().name());
            Assert.assertEquals((Object)Iterators.asSet((Object[])new String[]{this.propertyKey}), (Object)Iterables.asSet((Iterable)constraint.getPropertyKeys()));
            tx.success();
        }
    }

    @Test
    public void shouldListAddedConstraintsByLabel() {
        ConstraintDefinition constraint1 = this.createUniquenessConstraint(this.label, this.propertyKey);
        this.createUniquenessConstraint(Labels.MY_OTHER_LABEL, this.propertyKey);
        MatcherAssert.assertThat(Neo4jMatchers.getConstraints(this.db, this.label), Neo4jMatchers.containsOnly(constraint1));
    }

    @Test
    public void shouldListAddedConstraints() {
        ConstraintDefinition constraint1 = this.createUniquenessConstraint(Labels.MY_LABEL, this.propertyKey);
        ConstraintDefinition constraint2 = this.createUniquenessConstraint(Labels.MY_OTHER_LABEL, this.propertyKey);
        MatcherAssert.assertThat(Neo4jMatchers.getConstraints(this.db), Neo4jMatchers.containsOnly(constraint1, constraint2));
    }

    @Test
    public void shouldDropUniquenessConstraint() {
        ConstraintDefinition constraint = this.createUniquenessConstraint(this.label, this.propertyKey);
        this.dropConstraint(this.db, constraint);
        MatcherAssert.assertThat(Neo4jMatchers.getConstraints(this.db, this.label), Neo4jMatchers.isEmpty());
    }

    @Test
    public void addingConstraintWhenIndexAlreadyExistsGivesNiceError() {
        Neo4jMatchers.createIndex(this.db, this.label, this.propertyKey);
        try {
            this.createUniquenessConstraint(this.label, this.propertyKey);
            Assert.fail((String)"Expected exception to be thrown");
        }
        catch (ConstraintViolationException e) {
            Assert.assertEquals((Object)"There already exists an index for label 'MY_LABEL' on property 'my_property_key'. A constraint cannot be created until the index has been dropped.", (Object)e.getMessage());
        }
    }

    @Test
    public void addingUniquenessConstraintWhenDuplicateDataExistsGivesNiceError() {
        try (Transaction transaction = this.db.beginTx();){
            this.db.createNode(new Label[]{this.label}).setProperty(this.propertyKey, (Object)"value1");
            this.db.createNode(new Label[]{this.label}).setProperty(this.propertyKey, (Object)"value1");
            transaction.success();
        }
        try {
            this.createUniquenessConstraint(this.label, this.propertyKey);
            Assert.fail((String)"Expected exception to be thrown");
        }
        catch (ConstraintViolationException e) {
            MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"Unable to create CONSTRAINT ON ( my_label:MY_LABEL ) ASSERT my_label.my_property_key IS UNIQUE"));
        }
    }

    @Test
    public void addingConstraintWhenAlreadyConstrainedGivesNiceError() {
        this.createUniquenessConstraint(this.label, this.propertyKey);
        try {
            this.createUniquenessConstraint(this.label, this.propertyKey);
            Assert.fail((String)"Expected exception to be thrown");
        }
        catch (ConstraintViolationException e) {
            Assert.assertEquals((Object)"Constraint already exists: CONSTRAINT ON ( my_label:MY_LABEL ) ASSERT my_label.my_property_key IS UNIQUE", (Object)e.getMessage());
        }
    }

    @Test
    public void addingIndexWhenAlreadyConstrained() {
        this.createUniquenessConstraint(this.label, this.propertyKey);
        try {
            Neo4jMatchers.createIndex(this.db, this.label, this.propertyKey);
            Assert.fail((String)"Expected exception to be thrown");
        }
        catch (ConstraintViolationException e) {
            Assert.assertEquals((Object)"Label 'MY_LABEL' and property 'my_property_key' have a unique constraint defined on them, so an index is already created that matches this.", (Object)e.getMessage());
        }
    }

    @Test
    public void addingIndexWhenAlreadyIndexed() {
        Neo4jMatchers.createIndex(this.db, this.label, this.propertyKey);
        try {
            Neo4jMatchers.createIndex(this.db, this.label, this.propertyKey);
            Assert.fail((String)"Expected exception to be thrown");
        }
        catch (ConstraintViolationException e) {
            Assert.assertEquals((Object)"There already exists an index for label 'MY_LABEL' on property 'my_property_key'.", (Object)e.getMessage());
        }
    }

    @Test
    public void addedUncommittedIndexesShouldBeVisibleWithinTheTransaction() {
        IndexDefinition indexA = Neo4jMatchers.createIndex(this.db, this.label, "a");
        this.createUniquenessConstraint(this.label, "b");
        try (Transaction tx = this.db.beginTx();){
            MatcherAssert.assertThat((Object)Iterables.count((Iterable)this.db.schema().getIndexes(this.label)), (Matcher)Matchers.is((Object)2L));
            IndexDefinition indexC = this.db.schema().indexFor(this.label).on("c").create();
            MatcherAssert.assertThat((Object)Iterables.count((Iterable)this.db.schema().getIndexes(this.label)), (Matcher)Matchers.is((Object)3L));
            MatcherAssert.assertThat((Object)this.db.schema().getIndexState(indexA), (Matcher)Matchers.is((Object)Schema.IndexState.ONLINE));
            MatcherAssert.assertThat((Object)this.db.schema().getIndexState(indexC), (Matcher)Matchers.is((Object)Schema.IndexState.POPULATING));
        }
    }

    private void dropConstraint(GraphDatabaseService db, ConstraintDefinition constraint) {
        try (Transaction tx = db.beginTx();){
            constraint.drop();
            tx.success();
        }
    }

    private ConstraintDefinition createUniquenessConstraint(Label label, String prop) {
        try (Transaction tx = this.db.beginTx();){
            ConstraintDefinition constraint = this.db.schema().constraintFor(label).assertPropertyIsUnique(prop).create();
            tx.success();
            ConstraintDefinition constraintDefinition = constraint;
            return constraintDefinition;
        }
    }

    private void dropIndex(IndexDefinition index) {
        try (Transaction tx = this.db.beginTx();){
            index.drop();
            tx.success();
        }
    }

    private Node createNode(GraphDatabaseService db, String key, Object value, Label label) {
        try (Transaction tx = db.beginTx();){
            Node node = db.createNode(new Label[]{label});
            node.setProperty(key, value);
            tx.success();
            Node node2 = node;
            return node2;
        }
    }

    private static enum Labels implements Label
    {
        MY_LABEL,
        MY_OTHER_LABEL;

    }
}

