package darabonba.core;

import com.aliyun.core.annotation.EnumType;
import com.aliyun.core.annotation.NameInMap;
import com.aliyun.core.annotation.ParentIgnore;
import com.aliyun.core.annotation.Validation;
import com.aliyun.core.logging.ClientLogger;
import com.google.gson.Gson;
import darabonba.core.exception.TeaException;
import darabonba.core.exception.ValidateException;
import darabonba.core.utils.CommonUtil;
import darabonba.core.utils.ModelUtil;

import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.*;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

public class TeaModel {
    private static final ClientLogger logger = new ClientLogger(TeaModel.class);

    public Map<String, Object> toMap() {
        return changeToMap(this, true);
    }

    public static Map<String, Object> toMap(Object object) {
        return toMap(object, true);
    }

    public static Map<String, Object> toMap(Object object, boolean exceptStream) {
        Map<String, Object> map = new HashMap<String, Object>();
        if (object instanceof Map) {
            return (Map<String, Object>) object;
        }
        if (null == object || !TeaModel.class.isAssignableFrom(object.getClass())) {
            return map;
        }
        map = changeToMap(object, exceptStream);
        return map;
    }

    public Map<String, Object> toMap(boolean exceptStream) {
        return changeToMap(this, exceptStream);
    }

    private static Map<String, Object> changeToMap(Object object, boolean exceptStream) {
        HashMap<String, Object> map = new HashMap<String, Object>();
        try {
            for (Field field : object.getClass().getDeclaredFields()) {
                field.setAccessible(true);
                NameInMap anno = field.getAnnotation(NameInMap.class);
                String key;
                if (anno == null) {
                    key = field.getName();
                } else {
                    key = anno.value();
                }
                if (null != field.get(object) && List.class.isAssignableFrom(field.get(object).getClass())) {
                    List<Object> arrayField = (List<Object>) field.get(object);
                    List<Object> fieldList = new ArrayList<Object>();
                    for (int i = 0; i < arrayField.size(); i++) {
                        fieldList.add(ModelUtil.parseObject(arrayField.get(i)));
                    }
                    map.put(key, fieldList);
                } else if (null != field.get(object) && TeaModel.class.isAssignableFrom(field.get(object).getClass())) {
                    map.put(key, TeaModel.toMap(field.get(object), exceptStream));
                } else if (null != field.get(object) && field.get(object).getClass().getDeclaredAnnotation(EnumType.class) != null) {
                    Method method = field.get(object).getClass().getDeclaredMethod("getValue");
                    map.put(key, method.invoke(field.get(object)));
                } else if (null != field.get(object) && Map.class.isAssignableFrom(field.get(object).getClass())) {
                    Map<String, Object> valueMap = (Map<String, Object>) field.get(object);
                    Map<String, Object> result = new HashMap<String, Object>();
                    for (Map.Entry<String, Object> entry : valueMap.entrySet()) {
                        result.put(entry.getKey(), ModelUtil.parseObject(entry.getValue()));
                    }
                    map.put(key, result);
                } else if (exceptStream && null != field.get(object) && InputStream.class.isAssignableFrom(field.get(object).getClass())) {
                    continue;
                } else if (exceptStream && null != field.get(object) && OutputStream.class.isAssignableFrom(field.get(object).getClass())) {
                    continue;
                } else {
                    map.put(key, field.get(object));
                }
            }
        } catch (Exception e) {
            throw new TeaException(e.getMessage(), e);
        }
        return map;
    }

    private static Object buildObject(Object o, Class self, Type subType, String objectName) {
        Class valueClass = o.getClass();
        if (Map.class.isAssignableFrom(self) && Map.class.isAssignableFrom(valueClass)) {
            Map<String, Object> valueMap = (Map<String, Object>) o;
            Map<String, Object> result = new HashMap<String, Object>();
            for (Map.Entry<String, Object> entry : valueMap.entrySet()) {
                if (null == subType || subType instanceof WildcardType) {
                    result.put(entry.getKey(), entry.getValue());
                } else if (subType instanceof Class) {
                    result.put(entry.getKey(), buildObject(entry.getValue(), (Class) subType, null, objectName));
                } else {
                    ParameterizedType parameterizedType = (ParameterizedType) subType;
                    Type[] types = parameterizedType.getActualTypeArguments();
                    result.put(entry.getKey(), buildObject(entry.getValue(), (Class) parameterizedType.getRawType(), types[types.length - 1], objectName));
                }
            }
            return result;
        } else if (List.class.isAssignableFrom(self) && List.class.isAssignableFrom(valueClass)) {
            List<Object> valueList = (List<Object>) o;
            List<Object> result = new ArrayList<Object>();
            for (Object object : valueList) {
                if (null == subType || subType instanceof WildcardType) {
                    result.add(object);
                } else if (subType instanceof Class) {
                    result.add(buildObject(object, (Class) subType, null, objectName));
                } else {
                    ParameterizedType parameterizedType = (ParameterizedType) subType;
                    Type[] types = parameterizedType.getActualTypeArguments();
                    result.add(buildObject(object, (Class) parameterizedType.getRawType(), types[types.length - 1], objectName));
                }
            }
            return result;
        } else if (TeaModel.class.isAssignableFrom(self) && Map.class.isAssignableFrom(valueClass)) {
            try {
                Method method = self.getMethod("create");
                Object data = method.invoke(null);
                return TeaModel.toModel((Map<String, Object>) o, (TeaModel) data);
            } catch (Exception e) {
                throw new TeaException(e.getMessage(), e);
            }
        } else {
            return confirmType(self, o, objectName);
        }
    }

    private static Object buildPrefixObject(Object o, Class self, Type subType, String objectName) {
        Class valueClass = o.getClass();
        if (Map.class.isAssignableFrom(self) && Map.class.isAssignableFrom(valueClass)) {
            Map<String, Object> valueMap = (Map<String, Object>) o;
            Map<String, Object> result = new HashMap<String, Object>();
            for (Map.Entry<String, Object> entry : valueMap.entrySet()) {
                if (null == subType || subType instanceof WildcardType) {
                    result.put(entry.getKey(), entry.getValue());
                } else if (subType instanceof Class) {
                    result.put(entry.getKey(), buildPrefixObject(entry.getValue(), (Class) subType, null, objectName));
                } else {
                    ParameterizedType parameterizedType = (ParameterizedType) subType;
                    Type[] types = parameterizedType.getActualTypeArguments();
                    result.put(entry.getKey(), buildPrefixObject(entry.getValue(), (Class) parameterizedType.getRawType(), types[types.length - 1], objectName));
                }
            }
            return result;
        } else if (List.class.isAssignableFrom(self)) {
            List<Object> valueList = new ArrayList<>();
            if (List.class.isAssignableFrom(valueClass)) {
                valueList = (List<Object>) o;
            } else {
                valueList.add(o);
            }
            List<Object> result = new ArrayList<>();
            for (Object object : valueList) {
                if (null == subType || subType instanceof WildcardType) {
                    result.add(object);
                } else if (subType instanceof Class) {
                    result.add(buildPrefixObject(object, (Class) subType, null, objectName));
                } else {
                    ParameterizedType parameterizedType = (ParameterizedType) subType;
                    Type[] types = parameterizedType.getActualTypeArguments();
                    result.add(buildPrefixObject(object, (Class) parameterizedType.getRawType(), types[types.length - 1], objectName));
                }
            }
            return result;
        } else if (TeaModel.class.isAssignableFrom(self) && Map.class.isAssignableFrom(valueClass)) {
            try {
                Method method = self.getMethod("create");
                Object data = method.invoke(null);
                return TeaModel.adjustToModel((Map<String, Object>) o, (TeaModel) data);
            } catch (Exception e) {
                throw new TeaException(e.getMessage(), e);
            }
        } else {
            return confirmType(self, o, objectName);
        }
    }

    private static Type getType(Field field, int index) {
        ParameterizedType genericType = (ParameterizedType) field.getGenericType();
        Type[] actualTypeArguments = genericType.getActualTypeArguments();
        Type actualTypeArgument = actualTypeArguments[index];
        return actualTypeArgument;
    }


    /**
     * strict to model
     *
     * @param map
     * @param model
     * @param <T>
     * @return
     */
    public static <T extends TeaModel> T toModel(Map<String, ?> map, T model) {
        T result = model;
        Map<String, ?> mapCopy = null;
        for (Field field : result.getClass().getDeclaredFields()) {
            field.setAccessible(true);
            ParentIgnore parentIgnore = field.getAnnotation(ParentIgnore.class);
            mapCopy = map;
            if (parentIgnore != null) {
                String[] parents = parentIgnore.value().split(",");
                for (String parent : parents) {
                    mapCopy = (Map) mapCopy.get(parent);
                    if (mapCopy == null) {
                        break;
                    }
                }
            }
            NameInMap nameInMap = field.getAnnotation(NameInMap.class);
            String key;
            if (nameInMap == null) {
                key = field.getName();
            } else {
                key = nameInMap.value();
            }
            Object value = mapCopy != null ? mapCopy.get(key) : null;
            if (value == null) {
                continue;
            }
            result = setTeaModelField(result, field, value, result.getClass().getName() + "." + field.getName(), false);
        }
        return result;
    }

    /**
     * adjust to model
     *
     * @param map
     * @param model
     * @param <T>
     * @return
     */
    public static <T extends TeaModel> T adjustToModel(Map<String, ?> map, T model) {
        T result = model;
        Map<String, ?> mapCopy = null;
        for (Field field : result.getClass().getDeclaredFields()) {
            field.setAccessible(true);
            ParentIgnore parentIgnore = field.getAnnotation(ParentIgnore.class);
            mapCopy = map;
            if (parentIgnore != null) {
                String[] parents = parentIgnore.value().split(",");
                for (String parent : parents) {
                    if (mapCopy.get(parent) == null || !Map.class.isAssignableFrom(mapCopy.get(parent).getClass())) {
                        mapCopy = null;
                        break;
                    }
                    mapCopy = (Map) mapCopy.get(parent);
                }
            }
            NameInMap nameInMap = field.getAnnotation(NameInMap.class);
            String key;
            if (nameInMap == null) {
                key = field.getName();
            } else {
                key = nameInMap.value();
            }
            Object value = mapCopy != null ? mapCopy.get(key) : null;
            if (value == null) {
                continue;
            }
            result = setPrefixTeaModelField(result, field, value, result.getClass().getName() + "." + field.getName(), false);
        }
        return result;
    }

    private static <T extends TeaModel> T setTeaModelField(T model, Field field, Object value, String objectName, boolean userBuild) {
        try {
            Class<?> clazz = field.getType();
            Object resultValue = CommonUtil.parseNumber(value, clazz);
            T result = model;
            if (TeaModel.class.isAssignableFrom(clazz)) {
                Method method = clazz.getMethod("create");
                Object data = method.invoke(null);
                if (userBuild) {
                    field.set(result, TeaModel.build(TeaModel.toMap(resultValue, false), (TeaModel) data));
                } else if (!userBuild && Map.class.isAssignableFrom(resultValue.getClass())) {
                    field.set(result, TeaModel.toModel((Map<String, Object>) resultValue, (TeaModel) data));
                } else {
//                    if (StringUtils.isEmpty(resultValue)) {
//                        field.set(result, data);
//                    } else {
//                        field.set(result, resultValue);
//                    }
                    field.set(result, TeaModel.toModel(TeaModel.toMap(resultValue, false), (TeaModel) data));
                }
            } else if (Map.class.isAssignableFrom(clazz)) {
                boolean isUnmodifiableMap = false;
                try {
                    ((Map<?, ?>) resultValue).put(null, null);
                } catch (UnsupportedOperationException e) {
                    // UnmodifiableMap
                    field.set(result, resultValue);
                    isUnmodifiableMap = true;
                } catch (Exception e) {
                    // ignore
                }
                if (!isUnmodifiableMap) {
                    ((Map<?, ?>) resultValue).remove(null);
                    field.set(result, buildObject(resultValue, Map.class, getType(field, 1), objectName));
                }
            } else if (List.class.isAssignableFrom(clazz)) {
                field.set(result, buildObject(resultValue, List.class, getType(field, 0), objectName));
            } else if (Enum.class.isAssignableFrom(clazz)) {
                Class cl = Class.forName(clazz.getName());
                field.set(result, Enum.valueOf(cl, String.valueOf(resultValue).toUpperCase().replaceAll("-", "_")));
            } else {
                field.set(result, confirmType(clazz, resultValue, objectName));
            }
            return result;
        } catch (Exception e) {
            throw new TeaException(e.getMessage(), e);
        }
    }

    private static <T extends TeaModel> T setPrefixTeaModelField(T model, Field field, Object value, String objectName, boolean userBuild) {
        try {
            Class<?> clazz = field.getType();
            Object resultValue = CommonUtil.parseNumber(value, clazz);
            T result = model;
            if (TeaModel.class.isAssignableFrom(clazz)) {
                Method method = clazz.getMethod("create");
                Object data = method.invoke(null);
                if (userBuild) {
                    field.set(result, TeaModel.build(TeaModel.toMap(resultValue, false), (TeaModel) data));
                } else if (!userBuild && Map.class.isAssignableFrom(resultValue.getClass())) {
                    field.set(result, TeaModel.adjustToModel((Map<String, Object>) resultValue, (TeaModel) data));
                } else {
                    field.set(result, TeaModel.adjustToModel(TeaModel.toMap(resultValue, false), (TeaModel) data));
                }
            } else if (Map.class.isAssignableFrom(clazz)) {
                boolean isUnmodifiableMap = false;
                try {
                    ((Map<?, ?>) resultValue).put(null, null);
                } catch (UnsupportedOperationException e) {
                    // UnmodifiableMap
                    field.set(result, resultValue);
                    isUnmodifiableMap = true;
                } catch (Exception e) {
                    // ignore
                }
                if (!isUnmodifiableMap) {
                    ((Map<?, ?>) resultValue).remove(null);
                    field.set(result, buildPrefixObject(resultValue, Map.class, getType(field, 1), objectName));
                }
            } else if (List.class.isAssignableFrom(clazz)) {
                field.set(result, buildPrefixObject(resultValue, List.class, getType(field, 0), objectName));
            } else if (Enum.class.isAssignableFrom(clazz)) {
                Class cl = Class.forName(clazz.getName());
                field.set(result, Enum.valueOf(cl, String.valueOf(resultValue).toUpperCase().replaceAll("-", "_")));
            } else {
                field.set(result, confirmType(clazz, resultValue, objectName));
            }
            return result;
        } catch (Exception e) {
            throw new TeaException(e.getMessage(), e);
        }
    }

    @SuppressWarnings("unchecked")
    public static <T extends TeaModel> T build(Map<String, ?> map, T model) {
        T result = model;
        for (Field field : model.getClass().getDeclaredFields()) {
            field.setAccessible(true);
            String key = field.getName();
            Object value = map.get(key);
            if (value == null) {
                NameInMap anno = field.getAnnotation(NameInMap.class);
                if (null == anno) {
                    continue;
                }
                key = anno.value();
                value = map.get(key);
                if (null == value) {
                    continue;
                }
            }
            result = setTeaModelField(result, field, value, result.getClass().getName() + "." + field.getName(), true);
        }
        return result;
    }

    public void validate() {
        Field[] fields = this.getClass().getDeclaredFields();
        Object object;
        Validation validation;
        String pattern;
        int maxLength;
        int minLength;
        double maximum;
        double minimum;
        boolean required;
        try {
            for (int i = 0; i < fields.length; i++) {
                fields[i].setAccessible(true);
                object = fields[i].get(this);
                validation = fields[i].getAnnotation(Validation.class);
                if (null != validation) {
                    required = validation.required();
                } else {
                    required = false;
                }
                if (required && CommonUtil.isUnset(object)) {
                    throw new ValidateException("Field " + fields[i].getName() + " is required");
                }
                if (!CommonUtil.isUnset(validation) && !CommonUtil.isUnset(object)) {
                    pattern = validation.pattern();
                    maxLength = validation.maxLength();
                    minLength = validation.minLength();
                    maximum = validation.maximum();
                    minimum = validation.minimum();
                    if (!"".equals(pattern) || maxLength > 0 || minLength > 0 || maximum != Double.MAX_VALUE || minimum != Double.MIN_VALUE) {
                        determineType(fields[i].getType(), object, pattern, maxLength, minLength, maximum, minimum, fields[i].getName());
                    }
                }
            }
        } catch (Exception e) {
            throw new ValidateException(e.getMessage(), e);
        }
    }

    private void determineType(Class clazz, Object object, String pattern, int maxLength, int minLength, double maximum, double minimum, String fieldName) {
        if (Map.class.isAssignableFrom(clazz)) {
            validateMap(pattern, maxLength, minLength, maximum, minimum, (Map<String, Object>) object, fieldName);
        } else if (TeaModel.class.isAssignableFrom(clazz)) {
            ((TeaModel) object).validate();
        } else if (List.class.isAssignableFrom(clazz)) {
            List<?> list = (List<?>) object;
            for (int j = 0; j < list.size(); j++) {
                determineType(list.get(j).getClass(), list.get(j), pattern, maxLength, minLength, maximum, minimum, fieldName);
            }
        } else if (clazz.isArray()) {
            Object[] objects = (Object[]) object;
            for (int j = 0; j < objects.length; j++) {
                determineType(clazz.getComponentType(), objects[j], pattern, maxLength, minLength, maximum, minimum, fieldName);
            }
        } else if (Number.class.isAssignableFrom(clazz)) {
            double value = Double.valueOf(object.toString());
            if (value > maximum) {
                throw new ValidateException(this.getClass().getName() + "." + fieldName + " exceeds the maximum");
            }
            if (value < minimum) {
                throw new ValidateException(this.getClass().getName() + "." + fieldName + " less than minimum");
            }
        } else {
            String value = String.valueOf(object);
            if (maxLength > 0 && value.length() > maxLength) {
                throw new ValidateException(this.getClass().getName() + "." + fieldName + " exceeds the maximum length");
            }
            if (minLength > 0 && value.length() < minLength) {
                throw new ValidateException(this.getClass().getName() + "." + fieldName + " less than minimum length");
            }
            if (!"".equals(pattern) && !Pattern.matches(pattern, value)) {
                throw new ValidateException(this.getClass().getName() + "." + fieldName + " regular match failed");
            }
        }
    }

    private void validateMap(String pattern, int maxLength, int minLength, double maximum, double minimum, Map<String, Object> map, String fieldName) {
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            if (!CommonUtil.isUnset(entry.getValue())) {
                determineType(entry.getValue().getClass(), entry.getValue(), pattern, maxLength, minLength, maximum, minimum, fieldName);
            }
        }
    }

    public static Map<String, Object> buildMap(TeaModel teaModel) {
        if (null == teaModel) {
            return null;
        } else {
            return teaModel.toMap();
        }
    }

    public static void validateParams(TeaModel teaModel, String paramName) {
        if (null == teaModel) {
            throw new ValidateException("parameter " + paramName + " is not allowed as null");
        }
        teaModel.validate();
    }

    public static Object confirmType(Class expect, Object object) {
        return confirmType(expect, object, "unknown");
    }

    public static Object confirmType(Class expect, Object object, String objectName) {
        BigDecimal bigDecimal;
        if (String.class.isAssignableFrom(expect)) {
            if (object instanceof Number || object instanceof Boolean) {
                logger.info("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                        objectName, String.class.getName(), object.getClass().getName(), object.toString());
                return object.toString();
            }
            if (object instanceof Map || object instanceof List) {
                logger.info("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                        objectName, String.class.getName(), object.getClass().getName(), object.toString());
                return new Gson().toJson(object);
            }
        } else if (Boolean.class.isAssignableFrom(expect)) {
            if (object instanceof String) {
                logger.info("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                        objectName, Boolean.class.getName(), object.getClass().getName(), object.toString());
                return Boolean.parseBoolean(String.valueOf(object));
            } else if (object instanceof Number) {
                if (object.toString().equals("1")) {
                    logger.info("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                            objectName, Boolean.class.getName(), object.getClass().getName(), object.toString());
                    return true;
                } else if (object.toString().equals("0")) {
                    logger.info("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                            objectName, Boolean.class.getName(), object.getClass().getName(), object.toString());
                    return false;
                }
            }
        } else if (Integer.class.isAssignableFrom(expect)) {
            if (object instanceof String) {
                try {
                    Integer.parseInt(object.toString());
                } catch (NumberFormatException e) {
                    logger.warning("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                            objectName, Integer.class.getName(), object.getClass().getName(), object.toString());
                }
                bigDecimal = new BigDecimal(object.toString());
                return bigDecimal.intValue();
            }
            if (object instanceof Boolean) {
                logger.warning("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                        objectName, Integer.class.getName(), object.getClass().getName(), object.toString());
                return object.toString().equalsIgnoreCase("true") ? 1 : 0;
            }
            if (object instanceof Long) {
                if ((Long) object > Integer.MAX_VALUE) {
                    logger.warning("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                            objectName, Integer.class.getName(), object.getClass().getName(), object.toString());
                }
                bigDecimal = new BigDecimal(object.toString());
                return bigDecimal.intValue();
            }
            if (object instanceof Float || object instanceof Double) {
                bigDecimal = new BigDecimal(object.toString());
                logger.warning("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                        objectName, Integer.class.getName(), object.getClass().getName(), object.toString());
                return bigDecimal.intValue();
            }
        } else if (Long.class.isAssignableFrom(expect)) {
            if (object instanceof String || object instanceof Integer) {
                try {
                    Long.parseLong(object.toString());
                } catch (NumberFormatException e) {
                    logger.warning("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                            objectName, Long.class.getName(), object.getClass().getName(), object.toString());
                }
                bigDecimal = new BigDecimal(object.toString());
                return bigDecimal.longValue();
            }
            if (object instanceof Float || object instanceof Double) {
                bigDecimal = new BigDecimal(object.toString());
                logger.warning("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                        objectName, Long.class.getName(), object.getClass().getName(), object.toString());
                return bigDecimal.longValue();
            }
        } else if (Float.class.isAssignableFrom(expect)) {
            if (object instanceof String) {
                try {
                    Float.parseFloat(object.toString());
                } catch (NumberFormatException e) {
                    logger.warning("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                            objectName, Float.class.getName(), object.getClass().getName(), object.toString());
                }
                bigDecimal = new BigDecimal(object.toString());
                return bigDecimal.floatValue();
            }
            if (object instanceof Double) {
                if ((Double) object > Float.MAX_VALUE) {
                    logger.warning("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                            objectName, Float.class.getName(), object.getClass().getName(), object.toString());
                }
                bigDecimal = new BigDecimal(object.toString());
                return bigDecimal.floatValue();
            }
            if (object instanceof Integer || object instanceof Long) {
                bigDecimal = new BigDecimal(object.toString());
                logger.warning("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                        objectName, Float.class.getName(), object.getClass().getName(), object.toString());
                return bigDecimal.floatValue();
            }
        } else if (Double.class.isAssignableFrom(expect)) {
            if (object instanceof String || object instanceof Float) {
                try {
                    Double.parseDouble(object.toString());
                } catch (NumberFormatException e) {
                    logger.warning("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                            objectName, Double.class.getName(), object.getClass().getName(), object.toString());
                }
                bigDecimal = new BigDecimal(object.toString());
                return bigDecimal.doubleValue();
            }
            if (object instanceof Integer || object instanceof Long) {
                bigDecimal = new BigDecimal(object.toString());
                logger.warning("[{}] There are some cast events happening. expect: {}, but: {}, value: {}.",
                        objectName, Double.class.getName(), object.getClass().getName(), object.toString());
                return bigDecimal.doubleValue();
            }
        }
        return object;
    }
}
