package darabonba.core.policy.retry.backoff;

import darabonba.core.policy.retry.RetryPolicyContext;
import darabonba.core.policy.retry.RetryUtil;
import java.time.Duration;

public interface BackoffStrategy {

    /**
     * Max permitted retry times. To prevent exponentialDelay from overflow, there must be 2 ^ retriesAttempted
     * <= 2 ^ 31 - 1, which means retriesAttempted <= 30, so that is the ceil for retriesAttempted.
     */
    int RETRIES_ATTEMPTED_CEILING = (int) Math.floor(Math.log(Integer.MAX_VALUE) / Math.log(2));

    Duration computeDelayBeforeNextRetry(RetryPolicyContext context);

    default int calculateExponentialDelay(int retriesAttempted, Duration baseDelay, Duration maxBackoffTime) {
        int cappedRetries = Math.min(retriesAttempted, RETRIES_ATTEMPTED_CEILING);
        return (int) Math.min(baseDelay.multipliedBy(1L << cappedRetries).toMillis(), maxBackoffTime.toMillis());
    }

    static BackoffStrategy defaultStrategy() {
        return FullJitterBackoffStrategy.builder()
                                        .baseDelay(RetryUtil.BASE_DELAY)
                                        .maxBackoffTime(RetryUtil.MAX_BACKOFF)
                                        .build();
    }

    static BackoffStrategy defaultThrottlingStrategy() {
        return EqualJitterBackoffStrategy.builder()
                                         .baseDelay(RetryUtil.THROTTLED_BASE_DELAY)
                                         .maxBackoffTime(RetryUtil.MAX_BACKOFF)
                                         .build();
    }

    static BackoffStrategy none() {
        return FixedDelayBackoffStrategy.create(Duration.ofMillis(1));
    }
}
