package darabonba.core.utils;

import com.aliyun.core.utils.StringUtils;
import darabonba.core.TeaModel;
import darabonba.core.exception.TeaException;
import darabonba.core.TeaPair;

import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;

import static com.aliyun.core.utils.EncodeUtil.percentEncode;

public final class CommonUtil {

    public static boolean isUnset(Object object) {
        return null == object;
    }

    /**
     * If inputValue is not null, return it or return defaultValue
     *
     * @param inputValue   users input value
     * @param defaultValue default value
     * @return the final result
     */
    public static Object defaultAny(Object inputValue, Object defaultValue) {
        if (inputValue == null) {
            return defaultValue;
        }
        return inputValue;
    }

    /**
     * Assert a value, if it is a map,  it, otherwise throws
     *
     * @return the map value
     */
    public static Map<String, Object> assertAsMap(Object object) {
        if (null != object && Map.class.isAssignableFrom(object.getClass())) {
            return (Map<String, Object>) object;
        }
        throw new TeaException("The value is not a map", new RuntimeException("The value is not a object"));
    }

    /**
     * Assert a value, if it is a array, return it, otherwise throws
     *
     * @return the array
     */
    public static List<Object> assertAsArray(Object object) {
        if (null != object && List.class.isAssignableFrom(object.getClass())) {
            return (List<Object>) object;
        }
        throw new TeaException("The value is not a array", new RuntimeException("The value is not a array"));
    }

    /**
     * Assert a value, if it is a readable, return it, otherwise throws
     *
     * @return the readable value
     */
    public static InputStream assertAsReadable(Object value) {
        if (null != value && InputStream.class.isAssignableFrom(value.getClass())) {
            return (InputStream) value;
        }
        throw new TeaException("The value is not a readable", new RuntimeException("The value is not a readable"));
    }

    /**
     * Assert a value, if it is a bytes, return it, otherwise throws
     *
     * @return the bytes value
     */
    public static byte[] assertAsBytes(Object object) {
        if (object instanceof byte[]) {
            return (byte[]) object;
        }
        throw new TeaException("The value is not a byteArray", new RuntimeException("The value is not a byteArray"));
    }

    /**
     * Assert a value, if it is a number, return it, otherwise throws
     *
     * @return the number value
     */
    public static Number assertAsNumber(Object object) {
        if (object instanceof Number) {
            return (Number) object;
        }
        throw new TeaException("The value is not a Number", new RuntimeException("The value is not a Number"));
    }

    /**
     * Assert a value, if it is a string, return it, otherwise throws
     *
     * @return the string value
     */
    public static String assertAsString(Object object) {
        if (object instanceof String) {
            return (String) object;
        }
        throw new TeaException("The value is not a String", new RuntimeException("The value is not a String"));
    }

    /**
     * Assert a value, if it is a boolean, return it, otherwise throws
     *
     * @return the boolean value
     */
    public static Boolean assertAsBoolean(Object object) {
        try {
            return (Boolean) object;
        } catch (Exception e) {
            throw new TeaException("The value is not a Boolean", e);
        }
    }

    /**
     * Generate a nonce string
     *
     * @return the nonce string
     */
    public static String getNonce() {
        StringBuffer uniqueNonce = new StringBuffer();
        UUID uuid = UUID.randomUUID();
        uniqueNonce.append(uuid.toString());
        uniqueNonce.append(System.currentTimeMillis());
        uniqueNonce.append(Thread.currentThread().getId());
        return uniqueNonce.toString();
    }

    /**
     * Get an UTC format string by current date, e.g. 'Thu, 06 Feb 2020 07:32:54 GMT'
     *
     * @return the UTC format string
     */
    public static String getDateUTCString() {
        SimpleDateFormat df = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss z", Locale.US);
        df.setTimeZone(new SimpleTimeZone(0, "GMT"));
        return df.format(new Date());
    }

    /**
     * If the code after 300, return true, or return false
     *
     * @return boolean
     */
    public static boolean isNot2xx(Number code) {
        if (null == code) {
            return false;
        }
        return code.intValue() >= 300 ? true : false;
    }

    /**
     * If the code between 200 and 300, return true, or return false
     *
     * @return boolean
     */
    public static boolean is2xx(Number code) {
        if (null == code) {
            return false;
        }
        return code.intValue() >= 200 && code.intValue() < 300 ? true : false;
    }

    /**
     * If the code between 300 and 400, return true, or return false
     *
     * @return boolean
     */
    public static boolean is3xx(Number code) {
        if (null == code) {
            return false;
        }
        return code.intValue() >= 300 && code.intValue() < 400 ? true : false;
    }

    /**
     * If the code between 400 and 500, return true, or return false
     *
     * @return boolean
     */
    public static boolean is4xx(Number code) {
        if (null == code) {
            return false;
        }
        return code.intValue() >= 400 && code.intValue() < 500 ? true : false;
    }

    /**
     * If the code between 500 and 600, return true, or return false
     *
     * @return boolean
     */
    public static boolean is5xx(Number code) {
        if (null == code) {
            return false;
        }
        return code.intValue() >= 500 && code.intValue() < 600 ? true : false;
    }

    @SuppressWarnings("unchecked")
    public static <T> Map<String, T> buildMap(TeaPair... pairs) {
        Map<String, T> map = new HashMap<String, T>();
        for (int i = 0; i < pairs.length; i++) {
            TeaPair pair = pairs[i];
            map.put(pair.key, (T) pair.value);
        }
        return map;
    }

    public static <T> Map<String, T> merge(Class<T> t, Object... maps) {
        Map<String, T> out = new HashMap<String, T>();
        Map<String, ?> map = null;
        for (int i = 0; i < maps.length; i++) {
            if (null == maps[i]) {
                continue;
            }
            if (TeaModel.class.isAssignableFrom(maps[i].getClass())) {
                map = TeaModel.buildMap((TeaModel) maps[i]);
            } else {
                map = (Map<String, T>) maps[i];
            }
            Set<? extends Map.Entry<String, ?>> entries = map.entrySet();
            for (Map.Entry<String, ?> entry : entries) {
                if (null != entry.getValue()) {
                    out.put(entry.getKey(), (T) entry.getValue());
                }
            }
        }
        return out;
    }

    public static String getEncodePath(String path) throws UnsupportedEncodingException {
        if (StringUtils.isEmpty(path) || "/".equals(path)) {
            return path;
        }
        String[] strs = path.split("/");
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < strs.length; i++) {
            sb.append(percentEncode(strs[i]));
            sb.append("/");
        }
        return sb.deleteCharAt(sb.length() - 1).toString();
    }

    /**
     * Get timestamp
     *
     * @return the timestamp string
     */
    public static String getTimestamp() {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
        df.setTimeZone(new SimpleTimeZone(0, "UTC"));
        return df.format(new Date());
    }

    public static Object parseNumber(Object value, Class clazz) {
        BigDecimal bigDecimal;
        if (value instanceof Double && (clazz == Long.class || clazz == long.class)) {
            bigDecimal = new BigDecimal(value.toString());
            return bigDecimal.longValue();
        }
        if (value instanceof Double && (clazz == Integer.class || clazz == int.class)) {
            bigDecimal = new BigDecimal(value.toString());
            return bigDecimal.intValue();
        }
        if (value instanceof Double && (clazz == Float.class || clazz == float.class)) {
            bigDecimal = new BigDecimal(value.toString());
            return bigDecimal.floatValue();
        }
        return value;
    }
}

