/*
 * Copyright (C) 2016 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.apkzlib.sign;

import javax.annotation.Nonnull;

/**
 * Message digest algorithms.
 */
public enum DigestAlgorithm {
    /**
     * SHA-1 digest.
     * <p>
     * Android 2.3 (API Level 9) to 4.2 (API Level 17) (inclusive) do not support SHA-2
     * JAR signatures.
     * <p>
     * Moreover, platforms prior to API Level 18, without the additional
     * Digest-Algorithms attribute, only support SHA or SHA1 algorithm names in .SF and
     * MANIFEST.MF attributes.
     */
    SHA1("SHA1", "SHA-1"),

    /**
     * SHA-256 digest.
     */
    SHA256("SHA-256", "SHA-256");

    /**
     * API level which supports {@link #SHA256} with {@link SignatureAlgorithm#RSA}.
     */
    public static final int API_SHA_256_RSA = 18;

    /**
     * API level which supports {@link #SHA256} for all {@link SignatureAlgorithm}s.
     *
     * <p>Before that, SHA256 can only be used with RSA.
     */
    public static final int API_SHA_256_ALL_ALGORITHMS = 21;

    /**
     * Name of algorithm for message digest.
     */
    @Nonnull
    public final String messageDigestName;

    /**
     * Name of attribute in signature file with the manifest digest.
     */
    @Nonnull
    public final String manifestAttributeName;

    /**
     * Name of attribute in entry (both manifest and signature file) with the entry's digest.
     */
    @Nonnull
    public final String entryAttributeName;

    /**
     * Creates a digest algorithm.
     *
     * @param attributeName attribute name in the signature file
     * @param messageDigestName name of algorithm for message digest
     */
    DigestAlgorithm(@Nonnull String attributeName, @Nonnull String messageDigestName) {
        this.messageDigestName = messageDigestName;
        this.entryAttributeName = attributeName + "-Digest";
        this.manifestAttributeName = attributeName + "-Digest-Manifest";
    }

    /**
     * Finds the best digest algorithm applicable for a given SDK.
     *
     * @param minSdk the minimum SDK
     * @param signatureAlgorithm signature algorithm used
     * @return the best algorithm found
     */
    @Nonnull
    public static DigestAlgorithm findBest(
            int minSdk,
            @Nonnull SignatureAlgorithm signatureAlgorithm) {
        if (signatureAlgorithm == SignatureAlgorithm.RSA) {
            // PKCS #7 RSA signatures with SHA-256 are
            // supported only since API Level 18 (JB MR2).
            return minSdk >= API_SHA_256_RSA ? SHA256 : SHA1;
        } else {
            // PKCS #7 ECDSA and DSA signatures with SHA-256
            // are supported only since API Level 21 (Android L).
            return minSdk >= API_SHA_256_ALL_ALGORITHMS ? SHA256 : SHA1;
        }
    }
}
