/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.causalclustering.catchup.storecopy;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Supplier;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.rules.TestRule;
import org.neo4j.causalclustering.catchup.storecopy.StoreFiles;
import org.neo4j.causalclustering.identity.StoreId;
import org.neo4j.graphdb.mockfs.EphemeralFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.OpenMode;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.kernel.impl.store.MetaDataStore;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.impl.transaction.log.files.LogFilesBuilder;
import org.neo4j.test.rule.PageCacheRule;
import org.neo4j.test.rule.TestDirectory;
import org.neo4j.test.rule.fs.EphemeralFileSystemRule;

public class StoreFilesTest {
    protected TestDirectory testDirectory;
    protected Supplier<FileSystemAbstraction> fileSystemRule;
    protected EphemeralFileSystemRule hiddenFileSystemRule;
    protected PageCacheRule pageCacheRule;
    @Rule
    public RuleChain rules;
    private FileSystemAbstraction fs;
    private EphemeralFileSystemAbstraction pc;
    private PageCache pageCache;
    private StoreFiles storeFiles;
    private LogFiles logFiles;

    public StoreFilesTest() {
        this.createRules();
    }

    protected void createRules() {
        EphemeralFileSystemRule ephemeralFileSystemRule;
        this.testDirectory = TestDirectory.testDirectory(StoreFilesTest.class);
        this.fileSystemRule = ephemeralFileSystemRule = new EphemeralFileSystemRule();
        this.hiddenFileSystemRule = new EphemeralFileSystemRule();
        this.pageCacheRule = new PageCacheRule();
        this.rules = RuleChain.outerRule((TestRule)ephemeralFileSystemRule).around((TestRule)this.testDirectory).around((TestRule)this.hiddenFileSystemRule).around((TestRule)this.pageCacheRule);
    }

    @Before
    public void setUp() throws Exception {
        this.fs = this.fileSystemRule.get();
        this.pc = (EphemeralFileSystemAbstraction)this.hiddenFileSystemRule.get();
        this.pageCache = this.pageCacheRule.getPageCache((FileSystemAbstraction)this.pc);
        this.storeFiles = new StoreFiles(this.fs, this.pageCache);
        this.logFiles = LogFilesBuilder.logFilesBasedOnlyBuilder((File)this.testDirectory.directory(), (FileSystemAbstraction)this.fs).build();
    }

    private void createOnFileSystem(File file) throws IOException {
        this.createFile(this.fs, file);
    }

    private void createOnPageCache(File file) throws IOException {
        this.createFile(this.hiddenFileSystemRule.get(), file);
    }

    private void createFile(FileSystemAbstraction fs, File file) throws IOException {
        fs.mkdirs(file.getParentFile());
        fs.open(file, OpenMode.READ_WRITE).close();
    }

    protected File getBaseDir() {
        return new File(this.testDirectory.directory(), "dir");
    }

    @Test
    public void deleteMustRecursivelyRemoveFilesInGivenDirectory() throws Exception {
        File dir = this.getBaseDir();
        File a = new File(dir, "a");
        File b = new File(dir, "b");
        this.createOnFileSystem(a);
        this.createOnPageCache(b);
        Assert.assertTrue((boolean)this.fs.fileExists(a));
        Assert.assertTrue((boolean)this.pc.fileExists(b));
        this.storeFiles.delete(dir, this.logFiles);
        Assert.assertFalse((boolean)this.fs.fileExists(a));
        Assert.assertFalse((boolean)this.pc.fileExists(b));
    }

    @Test
    public void deleteMustNotDeleteIgnoredFiles() throws Exception {
        File dir = this.getBaseDir();
        File a = new File(dir, "a");
        File b = new File(dir, "b");
        File c = new File(dir, "c");
        File d = new File(dir, "d");
        this.createOnFileSystem(a);
        this.createOnFileSystem(c);
        this.createOnPageCache(b);
        this.createOnPageCache(d);
        FilenameFilter filter = (directory, name) -> !name.equals("c") && !name.equals("d");
        this.storeFiles = new StoreFiles(this.fs, this.pageCache, filter);
        this.storeFiles.delete(dir, this.logFiles);
        Assert.assertFalse((boolean)this.fs.fileExists(a));
        Assert.assertFalse((boolean)this.pc.fileExists(b));
        Assert.assertTrue((boolean)this.fs.fileExists(c));
        Assert.assertTrue((boolean)this.pc.fileExists(d));
    }

    @Test
    public void deleteMustNotDeleteFilesInIgnoredDirectories() throws Exception {
        File dir = this.getBaseDir();
        File ignore = new File(dir, "ignore");
        File a = new File(dir, "a");
        File b = new File(dir, "b");
        File c = new File(ignore, "c");
        File d = new File(ignore, "d");
        this.createOnFileSystem(a);
        this.createOnFileSystem(c);
        this.createOnPageCache(b);
        this.createOnPageCache(d);
        FilenameFilter filter = (directory, name) -> !name.startsWith("ignore");
        this.storeFiles = new StoreFiles(this.fs, this.pageCache, filter);
        this.storeFiles.delete(dir, this.logFiles);
        Assert.assertFalse((boolean)this.fs.fileExists(a));
        Assert.assertFalse((boolean)this.pc.fileExists(b));
        Assert.assertTrue((boolean)this.fs.fileExists(c));
        Assert.assertTrue((boolean)this.pc.fileExists(d));
    }

    @Test
    public void deleteMustSilentlyIgnoreMissingDirectories() throws Exception {
        File dir = this.getBaseDir();
        File sub = new File(dir, "sub");
        this.storeFiles.delete(sub, this.logFiles);
    }

    @Test
    public void mustMoveFilesToTargetDirectory() throws Exception {
        File base = this.getBaseDir();
        File src = new File(base, "src");
        File tgt = new File(base, "tgt");
        File a = new File(src, "a");
        File b = new File(src, "b");
        this.createOnFileSystem(a);
        this.createOnPageCache(b);
        this.createOnFileSystem(new File(tgt, ".fs-ignore"));
        this.createOnPageCache(new File(tgt, ".pc-ignore"));
        this.storeFiles.moveTo(src, tgt, this.logFiles);
        Assert.assertFalse((boolean)this.fs.fileExists(a));
        Assert.assertFalse((boolean)this.pc.fileExists(b));
        Assert.assertTrue((boolean)this.fs.fileExists(new File(tgt, "a")));
        Assert.assertTrue((boolean)this.pc.fileExists(new File(tgt, "b")));
    }

    @Test
    public void movedFilesMustRetainTheirRelativePaths() throws Exception {
        File base = this.getBaseDir();
        File src = new File(base, "src");
        File tgt = new File(base, "tgt");
        File dir = new File(src, "dir");
        File a = new File(dir, "a");
        File b = new File(dir, "b");
        this.createOnFileSystem(a);
        this.createOnPageCache(b);
        this.createOnFileSystem(new File(tgt, ".fs-ignore"));
        this.createOnPageCache(new File(tgt, ".pc-ignore"));
        this.storeFiles.moveTo(src, tgt, this.logFiles);
        Assert.assertFalse((boolean)this.fs.fileExists(a));
        Assert.assertFalse((boolean)this.pc.fileExists(b));
        Assert.assertTrue((boolean)this.fs.fileExists(new File(new File(tgt, "dir"), "a")));
        Assert.assertTrue((boolean)this.pc.fileExists(new File(new File(tgt, "dir"), "b")));
    }

    @Test
    public void moveMustIgnoreFilesFilteredOut() throws Exception {
        File base = this.getBaseDir();
        File src = new File(base, "src");
        File a = new File(src, "a");
        File b = new File(src, "b");
        File ignore = new File(src, "ignore");
        File c = new File(ignore, "c");
        File d = new File(ignore, "d");
        File tgt = new File(base, "tgt");
        this.createOnFileSystem(a);
        this.createOnPageCache(b);
        this.createOnFileSystem(c);
        this.createOnPageCache(d);
        this.createOnFileSystem(new File(tgt, ".fs-ignore"));
        this.createOnPageCache(new File(tgt, ".pc-ignore"));
        FilenameFilter filter = (directory, name) -> !name.startsWith("ignore");
        this.storeFiles = new StoreFiles(this.fs, this.pageCache, filter);
        this.storeFiles.moveTo(src, tgt, this.logFiles);
        Assert.assertFalse((boolean)this.fs.fileExists(a));
        Assert.assertFalse((boolean)this.pc.fileExists(b));
        Assert.assertTrue((boolean)this.fs.fileExists(c));
        Assert.assertTrue((boolean)this.pc.fileExists(d));
        Assert.assertTrue((boolean)this.fs.fileExists(new File(tgt, "a")));
        Assert.assertTrue((boolean)this.pc.fileExists(new File(tgt, "b")));
    }

    @Test
    public void isEmptyMustFindFilesBothOnFileSystemAndPageCache() throws Exception {
        File dir = this.getBaseDir();
        File ignore = new File(dir, "ignore");
        File a = new File(dir, "a");
        File b = new File(dir, "b");
        File c = new File(dir, "c");
        File d = new File(dir, "d");
        this.createOnFileSystem(a);
        this.createOnFileSystem(c);
        this.createOnFileSystem(ignore);
        this.createOnPageCache(b);
        this.createOnPageCache(d);
        this.createOnPageCache(ignore);
        FilenameFilter filter = (directory, name) -> !name.startsWith("ignore");
        this.storeFiles = new StoreFiles(this.fs, this.pageCache, filter);
        List<File> filesOnFilesystem = Arrays.asList(a, c);
        List<File> fileOnFilesystem = Arrays.asList(a);
        List<File> filesOnPageCache = Arrays.asList(b, d);
        List<File> fileOnPageCache = Arrays.asList(b);
        List<File> ingore = Arrays.asList(ignore);
        Assert.assertFalse((boolean)this.storeFiles.isEmpty(dir, filesOnFilesystem));
        Assert.assertFalse((boolean)this.storeFiles.isEmpty(dir, fileOnFilesystem));
        Assert.assertFalse((boolean)this.storeFiles.isEmpty(dir, filesOnPageCache));
        Assert.assertFalse((boolean)this.storeFiles.isEmpty(dir, fileOnPageCache));
        Assert.assertTrue((boolean)this.storeFiles.isEmpty(dir, Collections.emptyList()));
        Assert.assertTrue((boolean)this.storeFiles.isEmpty(dir, ingore));
    }

    @Test
    public void mustReadStoreId() throws Exception {
        File dir = this.getBaseDir();
        File neostore = new File(dir, "neostore");
        ThreadLocalRandom rng = ThreadLocalRandom.current();
        long time = rng.nextLong();
        long randomNumber = rng.nextLong();
        long upgradeTime = rng.nextLong();
        long upgradeTransactionId = rng.nextLong();
        this.createOnPageCache(neostore);
        MetaDataStore.setRecord((PageCache)this.pageCache, (File)neostore, (MetaDataStore.Position)MetaDataStore.Position.TIME, (long)time);
        MetaDataStore.setRecord((PageCache)this.pageCache, (File)neostore, (MetaDataStore.Position)MetaDataStore.Position.RANDOM_NUMBER, (long)randomNumber);
        MetaDataStore.setRecord((PageCache)this.pageCache, (File)neostore, (MetaDataStore.Position)MetaDataStore.Position.STORE_VERSION, (long)rng.nextLong());
        MetaDataStore.setRecord((PageCache)this.pageCache, (File)neostore, (MetaDataStore.Position)MetaDataStore.Position.UPGRADE_TIME, (long)upgradeTime);
        MetaDataStore.setRecord((PageCache)this.pageCache, (File)neostore, (MetaDataStore.Position)MetaDataStore.Position.UPGRADE_TRANSACTION_ID, (long)upgradeTransactionId);
        StoreId storeId = this.storeFiles.readStoreId(dir);
        Assert.assertThat((Object)storeId.getCreationTime(), (Matcher)Matchers.is((Object)time));
        Assert.assertThat((Object)storeId.getRandomId(), (Matcher)Matchers.is((Object)randomNumber));
        Assert.assertThat((Object)storeId.getUpgradeTime(), (Matcher)Matchers.is((Object)upgradeTime));
        Assert.assertThat((Object)storeId.getUpgradeId(), (Matcher)Matchers.is((Object)upgradeTransactionId));
    }
}

