/*
 * Copyright (C) 2020 Baidu, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package com.baidubce.services.iotdmp.model.service.rulechain;

import lombok.AllArgsConstructor;
import lombok.Getter;

@AllArgsConstructor
@Getter
public enum BlinkTopicInfo {

    /**
     * 定义: 单条属性上报
     * 操作:
     * 1、服务端订阅
     * 2、规则引擎
     * 3、更新设备影子(reported)
     */
    PROPERTY_POST(
            "属性上报",
            "设备属性状态单次上报",
            "thing/%s/%s/property/post",
            MessageType.DEVICE_MESSAGE,
            BlinkDataType.DEVICE_MESSAGE,
            BlinkDataPermission.UPWARD),
    /**
     * 定义: 属性批量上报
     * 操作:
     * 1、服务端订阅
     * 2、规则引擎
     */
    PROPERTY_BATCH(
            "属性上报",
            "设备属性状态批量上报",
            "thing/%s/%s/property/batch",
            MessageType.DEVICE_MESSAGE,
            BlinkDataType.DEVICE_MESSAGE,
            BlinkDataPermission.UPWARD),
    /**
     * 定义: 属性批量上报(文件Url)
     * 操作:
     * 1、服务端订阅
     * 2、规则引擎
     */
    PROPERTY_UPLOAD(
            "属性上报",
            "设备属性批量上传",
            "thing/%s/%s/property/upload",
            MessageType.DEVICE_MESSAGE,
            BlinkDataType.DEVICE_MESSAGE,
            BlinkDataPermission.UPWARD),
    /**
     * 定义: 更新设备期望值
     * 操作:
     * 1、服务端订阅
     * 2、规则引擎
     * 3、更新设备影子(desired)
     */
    PROPERTY_DESIRED(
            "属性上报",
            "更新设备期望值",
            "thing/%s/%s/property/desired",
            MessageType.DEVICE_MESSAGE,
            BlinkDataType.DEVICE_MESSAGE,
            BlinkDataPermission.UPWARD),
    /**
     * 定义: 请求设备影子数据
     * 操作:
     * 1、服务端订阅
     * 2、规则引擎
     * 3、下发设备影子数据
     */
    PROPERTY_GET(
            "数据请求",
            "请求设备影子数据",
            "thing/%s/%s/property/get",
            MessageType.DEVICE_MESSAGE,
            BlinkDataType.DEVICE_MESSAGE,
            BlinkDataPermission.UPWARD),
    /**
     * 定义: 下发设备影子数据
     * 操作:
     * 1、数据下发
     */
    PROPERTY_SNAPSHOT(
            "数据下发",
            "设备影子数据下发",
            "thing/%s/%s/property/snapshot",
            MessageType.DEVICE_MESSAGE,
            BlinkDataType.DEVICE_MESSAGE,
            BlinkDataPermission.DOWNWARD),
    /**
     * 定义: 设备影子数据与期望值差异
     * 操作:
     * 1、数据下发
     */
    PROPERTY_DELTA(
            "数据下发",
            "设备影子数据与期望值差异",
            "thing/%s/%s/property/delta",
            MessageType.DEVICE_MESSAGE,
            BlinkDataType.DEVICE_MESSAGE,
            BlinkDataPermission.SYSTEM),
    /**
     * 定义: 清空设备影子数据(reported)
     * 操作:
     * 1、服务端订阅
     * 2、规则引擎
     * 3、更新设备影子(reported)
     */
    PROPERTY_DELETE(
            "数据请求",
            "清空设备影子数据",
            "thing/%s/%s/property/delete",
            MessageType.DEVICE_MESSAGE,
            BlinkDataType.DEVICE_MESSAGE,
            BlinkDataPermission.UPWARD),
    /**
     * 定义: 单条事件上报
     * 操作:
     * 1、服务端订阅
     * 2、规则引擎
     */
    EVENT_POST(
            "事件上报",
            "设备事件单次上报",
            "thing/%s/%s/event/post",
            MessageType.DEVICE_MESSAGE,
            BlinkDataType.DEVICE_MESSAGE,
            BlinkDataPermission.UPWARD),
    /**
     * 定义: 事件批量上报
     * 操作:
     * 1、服务端订阅
     * 2、规则引擎
     */
    EVENT_BATCH(
            "事件上报",
            "设备事件批量上报",
            "thing/%s/%s/event/batch",
            MessageType.DEVICE_MESSAGE,
            BlinkDataType.DEVICE_MESSAGE,
            BlinkDataPermission.UPWARD),
    /**
     * 定义: 事件批量上报(文件Url)
     * 操作:
     * 1、服务端订阅
     * 2、规则引擎
     */
    EVENT_UPLOAD(
            "事件上报",
            "设备事件批量上传",
            "thing/%s/%s/event/upload",
            MessageType.DEVICE_MESSAGE,
            BlinkDataType.DEVICE_MESSAGE,
            BlinkDataPermission.UPWARD),
    /**
     * 定义: 设备可写属性更新
     * 操作:
     * 1、数据下发
     */
    PROPERTY_INVOKE(
            "属性下发",
            "设备可写属性更新",
            "thing/%s/%s/property/invoke",
            MessageType.DEVICE_MESSAGE,
            BlinkDataType.DEVICE_MESSAGE,
            BlinkDataPermission.DOWNWARD),
    /**
     * 定义: 设备服务调用
     * 操作:
     * 1、数据下发
     */
    COMMAND_INVOKE(
            "服务调用",
            "设备服务调用",
            "thing/%s/%s/command/invoke",
            MessageType.DEVICE_MESSAGE,
            BlinkDataType.DEVICE_MESSAGE,
            BlinkDataPermission.DOWNWARD),
    /**
     * 定义: cloud to device, 设备调用云端服务, 云端返回响应结果
     * 操作:
     * 1、数据下发
     */
    CLOUD_TO_DEVICE(
            "云端响应",
            "设备端调用云端响应",
            "thing/%s/%s/response/c2d",
            MessageType.DEVICE_MESSAGE,
            BlinkDataType.DEVICE_MESSAGE,
            BlinkDataPermission.DOWNWARD),
    /**
     * 定义: device to cloud, 云端调用设备服务, 设备返回响应结果
     * 操作:
     * 1、服务端订阅
     * 2、规则引擎
     */
    DEVICE_TO_CLOUD(
            "设备响应",
            "云端调用设备端响应",
            "thing/%s/%s/response/d2c",
            MessageType.DEVICE_MESSAGE,
            BlinkDataType.DEVICE_MESSAGE,
            BlinkDataPermission.UPWARD),

    /**
     * 定义: 设备生命周期变更
     * 操作:
     * 1、服务端订阅
     * 2、规则引擎
     */
    LIFECYCLE_UPDATE(
            "生命周期",
            "设备生命周期变更",
            "thing/%s/%s/lifecycle/update",
            MessageType.DEVICE_STATE_CHANGE,
            BlinkDataType.DEVICE_STATE_CHANGE,
            BlinkDataPermission.SYSTEM);

    private String commonName;
    private String description;
    private String pattern;
    private MessageType messageType;
    private BlinkDataType dataType;
    private BlinkDataPermission permission;

    public boolean isSourceable() {
        return permission.equals(BlinkDataPermission.UPWARD)
                || permission.equals(BlinkDataPermission.SYSTEM);
    }

    public boolean isSinkable() {
        return permission.equals(BlinkDataPermission.DOWNWARD);
    }
}
